/*
    Copyright 2011 Jean-Marc Alliot

    This file is part of the mpfi/mpfr bindings.

    The mpfi/mpfr bindings is free software: 
    you can redistribute it and/or modify it under the terms of 
    the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The mpfi/mpfr bindings is distributed in the hope that it will be 
    useful,but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public 
    License along with the mpfi/mpfr bindings.  
    If not, see <http://www.gnu.org/licenses/>.
*/



#include <stdio.h>
#include <mpfr.h>
#include <mpfi.h>
#include <mpfi_io.h>

#include <caml/mlvalues.h>
#include <caml/memory.h>
#include <caml/alloc.h>
#include <caml/custom.h>

/* Encapsulation of opaque mpfi handles (of type mpfi_t *)
   as Caml custom blocks. */

void finalize_mpfi(value v);
int compare_mpfi(value vm1,value vm2);

static struct custom_operations mpfi_ops = {
  "alliot.fr.mpfi",
  finalize_mpfi,
  compare_mpfi,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

/* Accessing the mpfi_t * part of a Caml custom block */
#define Mpfi_val(v) (*((mpfi_t **) Data_custom_val(v)))
/* Accessing the mpfr_t * part of a Caml custom block */
#define Mpfr_val(v) (*((mpfr_t **) Data_custom_val(v)))

void finalize_mpfi(value v)
{
  mpfi_t *m=Mpfi_val(v);
  mpfi_clear(*m);
  free(m);
}

/* The compare function is a pain in the ass. The standard mpfi_cmp
function is not compatible with ocaml convention because it can return 0
even if arguments are different, so I had to do it this way 
I'm not sure the function behaves properly for intervals containing NaN
values*/

int compare_mpfi(value vm1,value vm2)
{
  mpfi_t *m1=Mpfi_val(vm1);
  mpfi_t *m2=Mpfi_val(vm2);
  int prec1 = mpfi_get_prec(*m1);
  int prec2 = mpfi_get_prec(*m2);
  int c;
  mpfr_t m1_l,m1_r,m2_l,m2_r;

  mpfr_init2(m1_l,prec1);
  mpfr_init2(m1_r,prec1);
  mpfr_init2(m2_l,prec2);
  mpfr_init2(m2_r,prec2);
  mpfi_get_left(m1_l,*m1);
  mpfi_get_right(m1_r,*m1);
  mpfi_get_left(m2_l,*m2);
  mpfi_get_right(m2_r,*m2);

  c=mpfr_cmp(m1_l,m2_l);
  if (c==0) c=mpfr_cmp(m1_r,m2_r);

  mpfr_clear(m1_l);
  mpfr_clear(m1_r);
  mpfr_clear(m2_l);
  mpfr_clear(m2_r);
  return Int_val(c);
}

value caml_mpfi_init()
{
  CAMLlocal1(v);
  mpfi_t *m;
  m=(mpfi_t *)malloc(sizeof(mpfi_t));
  mpfi_init(*m);
  v = alloc_custom(&mpfi_ops, sizeof(mpfi_t *), 0, 1);
  Mpfi_val(v) = m;
  return v;
}

value caml_mpfi_init2(value vd)
{
  CAMLlocal1(v);
  mpfi_t *m;
  m=(mpfi_t *)malloc(sizeof(mpfi_t));
  mpfi_init2(*m,Val_int(vd));
  v = alloc_custom(&mpfi_ops, sizeof(mpfi_t *), 0, 1);
  Mpfi_val(v) = m;
  return v;
}

void caml_mpfi_set_prec(value vm,value vi)
{
  mpfi_set_prec(*(Mpfi_val(vm)),Int_val(vi));
}

value caml_mpfi_get_prec(value vm)
{
  return Val_int (mpfi_get_prec(*(Mpfi_val(vm))));
}

value caml_mpfi_round_prec(value vm,value vi)
{
  return Val_int(mpfi_round_prec(*(Mpfi_val(vm)),Int_val(vi)));
}

value caml_mpfi_set(value vm1,value vm2)
{
  return Val_int(mpfi_set(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_set_d(value vm,value vd)
{
  return Val_int(mpfi_set_d(*(Mpfi_val(vm)),Double_val(vd)));
}

value caml_mpfi_set_str(value vm,value vs,value vd)
{
  return Val_int(mpfi_set_str(*(Mpfi_val(vm)),String_val(vs),Int_val(vd)));
}

value caml_mpfi_add(value vm1,value vm2,value vm3)
{
  return Val_int(mpfi_add(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),*(Mpfi_val(vm3))));
}

value caml_mpfi_sub(value vm1,value vm2,value vm3)
{
  return Val_int(mpfi_sub(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),*(Mpfi_val(vm3))));
}

value caml_mpfi_mul(value vm1,value vm2,value vm3)
{
  return Val_int(mpfi_mul(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),*(Mpfi_val(vm3))));
}

value caml_mpfi_div(value vm1,value vm2,value vm3)
{
  return Val_int(mpfi_div(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),*(Mpfi_val(vm3))));
}

value caml_mpfi_neg(value vm1,value vm2)
{
  return Val_int(mpfi_neg(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_inv(value vm1,value vm2)
{
  return Val_int(mpfi_inv(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_sqr(value vm1,value vm2)
{
  return Val_int(mpfi_sqr(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_sqrt(value vm1,value vm2)
{
  return Val_int(mpfi_sqrt(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_abs(value vm1,value vm2)
{
  return Val_int(mpfi_abs(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_log(value vm1,value vm2)
{
  return Val_int(mpfi_log(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_exp(value vm1,value vm2)
{
  return Val_int(mpfi_exp(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_exp2(value vm1,value vm2)
{
  return Val_int(mpfi_exp2(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_cos(value vm1,value vm2)
{
  return Val_int(mpfi_cos(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_sin(value vm1,value vm2)
{
  return Val_int(mpfi_sin(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_tan(value vm1,value vm2)
{
  return Val_int(mpfi_tan(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}


value caml_mpfi_acos(value vm1,value vm2)
{
  return Val_int(mpfi_acos(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_asin(value vm1,value vm2)
{
  return Val_int(mpfi_asin(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_atan(value vm1,value vm2)
{
  return Val_int(mpfi_atan(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_acosh(value vm1,value vm2)
{
  return Val_int(mpfi_acosh(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_asinh(value vm1,value vm2)
{
  return Val_int(mpfi_asinh(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_atanh(value vm1,value vm2)
{
  return Val_int(mpfi_atanh(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_cosh(value vm1,value vm2)
{
  return Val_int(mpfi_cosh(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_sinh(value vm1,value vm2)
{
  return Val_int(mpfi_sinh(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_tanh(value vm1,value vm2)
{
  return Val_int(mpfi_tanh(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_log1p(value vm1,value vm2)
{
  return Val_int(mpfi_log1p(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_expm1(value vm1,value vm2)
{
  return Val_int(mpfi_expm1(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_log2(value vm1,value vm2)
{
  return Val_int(mpfi_log2(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_log10(value vm1,value vm2)
{
  return Val_int(mpfi_log10(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_add_d(value vm1,value vm2,value vd)
{
  return Val_int(mpfi_add_d(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),Double_val(vd)));
}

value caml_mpfi_sub_d(value vm1,value vm2,value vd)
{
  return Val_int(mpfi_sub_d(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),Double_val(vd)));
}

value caml_mpfi_mul_d(value vm1,value vm2,value vd)
{
  return Val_int(mpfi_mul_d(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),Double_val(vd)));
}

value caml_mpfi_div_d(value vm1,value vm2,value vd)
{
  return Val_int(mpfi_div_d(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),Double_val(vd)));
}

value caml_mpfi_d_sub(value vm1,value vd,value vm2)
{
  return Val_int(mpfi_d_sub(*(Mpfi_val(vm1)),Double_val(vd),*(Mpfi_val(vm2))));
}

value caml_mpfi_d_div(value vm1,value vd,value vm2)
{
  return Val_int(mpfi_d_div(*(Mpfi_val(vm1)),Double_val(vd),*(Mpfi_val(vm2))));
}

value caml_mpfi_get_d(value vm1)
{
  return caml_copy_double(mpfi_get_d(*(Mpfi_val(vm1))));
}

value caml_mpfi_const_log2(value vm1)
{
  return Val_int(mpfi_const_log2(*(Mpfi_val(vm1))));
}

value caml_mpfi_const_pi(value vm1)
{
  return Val_int(mpfi_const_pi(*(Mpfi_val(vm1))));
}

value caml_mpfi_const_euler(value vm1)
{
  return Val_int(mpfi_const_euler(*(Mpfi_val(vm1))));
}

value caml_mpfi_cmp(value vm1,value vm2)
{
  return Val_int(mpfi_cmp(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_cmp_d(value vm1,value vd)
{
  return Val_int(mpfi_cmp_d(*(Mpfi_val(vm1)),Double_val(vd)));
}

value caml_mpfi_is_pos(value vm1)
{
  return Val_int(mpfi_is_pos(*(Mpfi_val(vm1))));
}

value caml_mpfi_is_strictly_pos(value vm1)
{
  return Val_int(mpfi_is_strictly_pos(*(Mpfi_val(vm1))));
}

value caml_mpfi_is_neg(value vm1)
{
  return Val_int(mpfi_is_neg(*(Mpfi_val(vm1))));
}

value caml_mpfi_is_strictly_neg(value vm1)
{
  return Val_int(mpfi_is_strictly_neg(*(Mpfi_val(vm1))));
}

value caml_mpfi_is_nonpos(value vm1)
{
  return Val_int(mpfi_is_nonpos(*(Mpfi_val(vm1))));
}

value caml_mpfi_is_zero(value vm1)
{
  return Val_int(mpfi_is_zero(*(Mpfi_val(vm1))));
}

value caml_mpfi_has_zero(value vm1)
{
  return Val_int(mpfi_has_zero(*(Mpfi_val(vm1))));
}

value caml_mpfi_nan_p(value vm1)
{
  return Val_int(mpfi_nan_p(*(Mpfi_val(vm1))));
}

value caml_mpfi_inf_p(value vm1)
{
  return Val_int(mpfi_inf_p(*(Mpfi_val(vm1))));
}

value caml_mpfi_bounded_p(value vm1)
{
  return Val_int(mpfi_bounded_p(*(Mpfi_val(vm1))));
}


value caml_mpfi_out_str(value vi1,value vi2,value vm)
{
  return Val_int(mpfi_out_str(stdout,Int_val(vi1),Int_val(vi2),*(Mpfi_val(vm))));
}

value caml_mpfi_get_left_d(value vm)
{
  mpfi_t *m=Mpfi_val(vm);
  int prec = mpfi_get_prec(*m);
  mpfr_t fr;
  double d;

  mpfr_init2(fr,prec);
  mpfi_get_left(fr,*m);

  d=mpfr_get_d(fr,MPFR_RNDD);
  mpfr_clear(fr);
  return caml_copy_double(d);
}

value caml_mpfi_get_right_d(value vm)
{
  mpfi_t *m=Mpfi_val(vm);
  int prec = mpfi_get_prec(*m);
  mpfr_t fr;
  double d;

  mpfr_init2(fr,prec);
  mpfi_get_right(fr,*m);

  d=mpfr_get_d(fr,MPFR_RNDD);
  mpfr_clear(fr);
  return caml_copy_double(d);
}

value caml_mpfi_get_left(value vr,value vm)
{
  mpfi_t *m=Mpfi_val(vm);
  mpfr_t *fr=Mpfr_val(vr);

  return Val_int(mpfi_get_left(*fr,*m));
}

value caml_mpfi_get_right(value vr,value vm)
{
  mpfi_t *m=Mpfi_val(vm);
  mpfr_t *fr=Mpfr_val(vr);

  return Val_int(mpfi_get_right(*fr,*m));
}

/* On ly available in mpfi 1.5.1
value caml_mpfi_cbrt(value vm1,value vm2)
{
  return Val_int(mpfi_cbrt(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_csc(value vm1,value vm2)
{
  return Val_int(mpfi_csc(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_sec(value vm1,value vm2)
{
  return Val_int(mpfi_sec(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_cot(value vm1,value vm2)
{
  return Val_int(mpfi_cot(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_csch(value vm1,value vm2)
{
  return Val_int(mpfi_csch(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_sech(value vm1,value vm2)
{
  return Val_int(mpfi_sech(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_coth(value vm1,value vm2)
{
  return Val_int(mpfi_coth(*(Mpfi_val(vm1)),*(Mpfi_val(vm2))));
}

value caml_mpfi_atan2(value vm1,value vm2,value vm3)
{
  return Val_int(mpfi_atan2(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),*(Mpfi_val(vm3))));
}

value caml_mpfi_const_catalan(value vm1)
{
  return Val_int(mpfi_const_catalan(*(Mpfi_val(vm1))));
}

value caml_mpfi_hypot(value vm1,value vm2,value vm3)
{
  return Val_int(mpfi_hypot(*(Mpfi_val(vm1)),*(Mpfi_val(vm2)),*(Mpfi_val(vm3))));
}

*/
